<?php
/**
 * Orvis functions and definitions.
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Orvis
 */

if ( ! function_exists( 'orvis_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function orvis_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on Orvis, use a find and replace
	 * to change 'orvis' to the name of your theme in all the template files.
	 */
	load_theme_textdomain( 'orvis', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	 */
	add_theme_support( 'post-thumbnails' );
	set_post_thumbnail_size( 924, 0, true );
	add_image_size( 'orvis-project-thumbnail', 671, 0, true );

	// Enable support for custom logo.
	add_theme_support( 'custom-logo', array(
		'height'      => 176,
		'width'       => 1248,
		'flex-height' => true,
		'flex-width'  => true,
	) );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus( array(
		'primary' => esc_html__( 'Primary Menu', 'orvis' ),
	) );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'search-form',
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) );

	// Set up the WordPress core custom background feature.
	add_theme_support( 'custom-background', apply_filters( 'orvis_custom_background_args', array(
		'default-color' => '303030',
		'default-image' => '',
	) ) );

	/*
	 * This theme styles the visual editor to resemble the theme style,
	 * specifically font, colors, icons, and column width.
	 */
	add_editor_style( array( 'editor-style.css', orvis_fonts_url() ) );
}
endif; // orvis_setup
add_action( 'after_setup_theme', 'orvis_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function orvis_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'orvis_content_width', 924 );
}
add_action( 'after_setup_theme', 'orvis_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function orvis_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Sidebar', 'orvis' ),
		'id'            => 'sidebar-1',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
}
add_action( 'widgets_init', 'orvis_widgets_init' );

if ( ! function_exists( 'orvis_fonts_url' ) ) :
/**
 * Register Google fonts for Orvis.
 *
 * Create your own orvis_fonts_url() function to override in a child theme.
 *
 * @return string Google fonts URL for the theme.
 */
function orvis_fonts_url() {
	$fonts_url = '';
	$fonts     = array();
	$subsets   = 'latin,latin-ext';

	/* translators: If there are characters in your language that are not supported by Karla, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== esc_html_x( 'on', 'Karla font: on or off', 'orvis' ) ) {
		$fonts[] = 'Karla:400,700,400italic,700italic';
	}

	if ( $fonts ) {
		$fonts_url = add_query_arg( array(
			'family' => urlencode( implode( '|', $fonts ) ),
			'subset' => urlencode( $subsets ),
		), 'https://fonts.googleapis.com/css' );
	}

	return $fonts_url;
}
endif;

/**
 * Enqueue scripts and styles.
 */
function orvis_scripts() {
	wp_enqueue_style( 'orvis-fonts', orvis_fonts_url(), array(), null );

	wp_enqueue_style( 'orvis-style', get_stylesheet_uri() );

	wp_enqueue_script( 'orvis-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array(), '20151231', true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

	wp_enqueue_script( 'orvis-script', get_template_directory_uri() . '/js/functions.js', array( 'jquery' ), '20151231', true );

	wp_localize_script( 'orvis-script', 'screenReaderText', array(
		'expand'   => esc_html__( 'expand child menu', 'orvis' ),
		'collapse' => esc_html__( 'collapse child menu', 'orvis' ),
	) );
}
add_action( 'wp_enqueue_scripts', 'orvis_scripts' );

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom functions that act independently of the theme templates.
 */
require get_template_directory() . '/inc/extras.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
require get_template_directory() . '/inc/jetpack.php';


/**
 * Load plugin enhancement file to display admin notices.
 */
require get_template_directory() . '/inc/plugin-enhancements.php';